# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidGlobalVars', 'Global:FormatEnumerationLimit')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $AdapterName = $Name
        $ModuleNames = $Module
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
                continue
            }

            $Local:IntelLogConfigurations = @{}
            $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
            $Local:IntelLogConfigModules = @{}
            $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"
            $Local:IntelLogConfigLevels = @{}
            $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

            if ($IntelLogConfigLevels.Count -eq 0)
            {
                $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
            }

            $DisplayConfigNames = @()
            if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Get" $a))
            {
                if ($script:ErrorMessagesGet.Count -gt 0)
                {
                    break
                }
                continue
            }

            $DisplayModules = @()
            $DisplayLevels = @()
            $DriverPassed = $null

            if ($DisplayConfigNames)
            {
                $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"
                if (!(ValidateMethodOutput $MethodOutput "Get"))
                {
                    continue
                }

                GetConfigLogs $DisplayConfigNames ([ref]$DisplayModules) ([ref]$DisplayLevels) $MethodOutput
            }
            else
            {
                $DriverPassed = $ModuleNames.Where({$DRIVER_MODULE_NAME -like $_})

                if (-Not $DriverPassed -and $SubModule)
                {
                    $script:ErrorMessagesGet += $Messages.InvalidParams
                    break
                }
                elseif ($SubModule)
                {
                    $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesGet)
                    if ($SubModuleSum -eq -1)
                    {
                        break
                    }
                }

                if ($DriverPassed -or -Not $ModuleNames)
                {
                    $LogConfigs += GetDriverLogConfig $DriverPassed $SubModuleSum $a ([ref]$script:ErrorMessagesGet)
                }

                $MethodOutput = $null
                if ($IntelLogConfigModules.Count -gt 0)
                {
                    $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"

                    if (!(ValidateMethodOutput $MethodOutput "Get"))
                    {
                        continue
                    }
                }

                # ValidateGetModuleParam will error if Driver is passed and not present here.
                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                ValidateGetModuleParam ([ref]$DisplayModules) ([ref]$DisplayLevels) $ModuleNames $MethodOutput $a
            }

            if ($DisplayModules -and $DisplayLevels)
            {
                $LogConfigs += GatherOutput $a $DisplayModules $DisplayLevels $DriverPassed
            }
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }
        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false,Position=2)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String]
    $Level = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $ModuleNames = $Module
        $LevelName = $Level
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:WarningMessagesSet = @()
        $script:ErrorMessagesSet = @()
        $LogConfigs = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        do
        {
            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesSet)
            $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))

            if ([string]::IsNullOrEmpty($ConfigurationNames) -and ([string]::IsNullOrEmpty($ModuleNames)))
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigOrModule
                break
            }

            foreach($a in $AdapterNames)
            {
                $TmpStatusMsg = CheckDeviceError $a
                if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                {
                    $script:ErrorMessagesSet += $TmpStatusMsg
                    continue
                }

                $DisplayModules = @()
                $DisplayLevels = @()
                $InputModulesId = [uint32[]]::new(64)
                $InputLevelId = [uint32[]]::new(64)

                $Local:IntelLogConfigurations = @{}
                $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
                $Local:IntelLogConfigModules = @{}
                $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"

                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                $ModuleCount = 0
                $DisplayConfigNames = @()
                if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Set" $a))
                {
                    break
                }

                $bDriverPassed = $false
                $InputConfigurationIDs = $null
                if ($DisplayConfigNames)
                {
                    $InputConfigurationIDs = $IntelLogConfigurations[$DisplayConfigNames]
                }
                else
                {
                    ValidateSetModuleParam ([ref]$InputModulesId) ([ref]$DisplayModules) $ModuleNames $a

                    if ($DisplayModules -contains $DRIVER_MODULE_NAME)
                    {
                        $bDriverPassed = $true
                    }
                    elseif ($SubModule)
                    {
                        $script:ErrorMessagesSet += $Messages.InvalidParams
                        break
                    }

                    $ModuleCount = $DisplayModules.Count
                }

                if ($ModuleCount -ne 0)
                {
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $ModuleCount $a $bDriverPassed))
                    {
                        break
                    }

                    if ($bDriverPassed)
                    {
                        # Driver module is not included in Wmi method params
                        $ModuleCount -= 1

                        if ($InputLevelId[0] -lt 3)
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidDriverLevel
                            break
                        }

                        if ($SubModule)
                        {
                            $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesSet)
                            if ($SubModuleSum -eq -1)
                            {
                                break
                            }
                        }

                        if (-not (SetSubModuleRegistryValue $a $SubModuleSum))
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                            break
                        }

                        $LevelName = GetLevelName($InputLevelId[0])

                        if (-Not (SetDriverLogLevel $a $LevelName))
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                            break
                        }
                    }

                    if ($ModuleCount -gt 0)
                    {
                        $params = @{Count    = [uint32]$ModuleCount;
                                ModuleID = $InputModulesId;
                                Loglevel = $InputLevelId}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogLevels" $params
                    }

                    if ($bDriverPassed)
                    {
                        # Driver is set, display it. Avoid call to Validate if no other modules passed.
                        if ($ModuleCount -eq 0 -or (ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules -SubModule $SubModuleSum
                        }
                        else
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule $SubModuleSum
                        }
                    }
                    elseif (ValidateMethodOutput $MethodOutput "Set")
                    {
                        $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules
                    }
                }
                elseif ($null -ne $InputConfigurationIDs)
                {
                    $count = 1
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $count $a))
                    {
                        break
                    }

                    $bContinue = $true
                    foreach ($ConfigID in $InputConfigurationIDs)
                    {
                        $params = @{ConfigurationID = [uint32]$ConfigID;
                                    Level = $InputLevelId[0]}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogConfiguration" $params

                        if (!(ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $bContinue = $false
                            break
                        }
                    }

                    if (!$bContinue)
                    {
                        break
                    }
                    $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Configuration $DisplayConfigNames
                }
            }
        } while ($false)
    }
    End
    {
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Start-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [ValidateNotNullOrEmpty()]
    [string]
    $Path = '',
    [parameter(Mandatory=$false)]
    [switch]
    $Force,
    [parameter(Mandatory=$false)]
    [switch]
    $Append
    )
    Begin
    {
        $SubModuleSum = 0xFFFF
        # Set LogmanLevel to Normal by default
        $LogmanLevel = 4
        $script:ErrorMessagesStart = @()
        $script:WarningMessagesStart = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesStart += $Messages.AdapterNotFound -f $Name
                break
            }

            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStart)
            $AdapterName = GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesStart)

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (-not (ValidateSingleAdapter $input $AdapterName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $TmpStatusMsg = CheckDeviceError $AdapterName
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesStart += $TmpStatusMsg
                break
            }

            $UseDefaultPath = $false
            if (([string]::IsNullOrEmpty($Path)))
            {
                $UseDefaultPath = $true
            }

            $LogmanGUID = GetLogmanGUID $AdapterName ([ref]$script:ErrorMessagesStart)
            $LogName = GetLogName $AdapterName ([ref]$script:ErrorMessagesStart)

            if ($null -eq $LogmanGUID -or $null -eq $LogName)
            {
                break
            }

            if (LogmanProcessRunning $AdapterName $LogName $LogmanGUID)
            {
                break
            }

            $LogmanPath = ''
            if (-not (ValidatePathParams ([ref]$LogmanPath) $UseDefaultPath $LogName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $DriverLogLevel = GetDriverLogLevel $AdapterName
            if ($DriverLogLevel)
            {
                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $AdapterName "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                # Driver log levels are normal levels value + 1
                $LogmanLevel = $IntelLogConfigLevels[$DriverLogLevel] + 1

                $LogMetadataName = "$LogName.txt"

                $LogMetadataPath = ''
                if (-Not $UseDefaultPath)
                {
                    if ([IO.Path]::GetExtension($Path) -eq ".txt")
                    {
                        $Path += ".txt"
                    }
                    else
                    {
                        $Path = [IO.Path]::ChangeExtension($Path, ".txt")
                    }
                }

                $iBeforeErrorCount = $script:ErrorMessagesStart.Count
                if (-not (ValidatePathParams ([ref]$LogMetadataPath) $UseDefaultPath $LogMetadataName ([ref]$script:ErrorMessagesStart)))
                {
                    if ($iBeforeErrorCount -lt $script:ErrorMessagesStart.Count -and $script:ErrorMessagesStart[-1] -eq $Messages.LogmanFileExists -f $AdapterName)
                    {
                        # Driver Metadata File is always replaced if it exists.
                        $script:ErrorMessagesStart = $script:ErrorMessagesStart | Select-Object -First ($script:ErrorMessagesStart.Count - 1)
                    }
                    else
                    {
                        break
                    }
                }

                $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
                if ($null -eq $AdapterRegistryPath)
                {
                    $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
                    break
                }

                $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
                if ($SubModuleRegistryValue)
                {
                    $SubModuleSum = $SubModuleRegistryValue.SubModule

                    if (-Not (CreateDriverLogMetadataFile $AdapterName))
                    {
                        break
                    }
                }
            }

            LogmanStart $LogmanPath $LogName $LogmanGUID $SubModuleSum $LogmanLevel
            if (ValidateLogmanStart $AdapterName)
            {
                $LogmanPath = Resolve-Path -Path $LogmanPath -ErrorAction SilentlyContinue
                Write-Output $LogmanPath
            }

            break
        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesStart)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStart)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Stop-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null
    )
    Begin
    {
        $script:ErrorMessagesStop = @()
        $script:WarningMessagesStop = @()
        $bErrorObtained = $false
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStop)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesStop))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesStop += $TmpStatusMsg
                continue
            }

            if (-Not $bErrorObtained)
            {
                $LogName = GetLogName $a ([ref]$script:ErrorMessagesStop)
            }
            else
            {
                $LogName = GetLogName $a ([ref]@())
            }

            if ($null -eq $LogName)
            {
                $bErrorObtained = $true
                continue
            }

            LogmanStop $LogName
            if (-Not $bErrorObtained)
            {
                if (-Not (ValidateLogmanStop $a ([ref]$script:ErrorMessagesStop)))
                {
                    $bErrorObtained = $true
                }
            }
        }
    }
    End
    {
        foreach ($WarningMessage in $script:WarningMessagesStop)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStop)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Disable-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesDisable = @()
        $script:WarningMessagesDisable = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesDisable)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesDisable))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesDisable += $TmpStatusMsg
                continue
            }

            $DriverConfig = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule @($SUB_MODULE_MAP.Keys)[0] -Level "Normal"
            if (-Not $DriverConfig)
            {
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiDisableFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Disable"))
                {
                    continue
                }
            }

            # Will pass Driver as module, so returns IntelDriverLogConfiguration
            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*" -SubModule @($SUB_MODULE_MAP.Keys)[0]
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesDisable)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesDisable)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Reset-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesReset = @()
        $script:WarningMessagesReset = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesReset)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesReset))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesReset += $TmpStatusMsg
                continue
            }

            # Set driver level back to Normal
            $null = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiResetFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Reset"))
                {
                    continue
                }
            }

            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*"
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesReset)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesReset)
        {
            Write-Error $ErrorMessage
        }
    }
}

Function GatherOutput($AdapterName, $DisplayModules, $DisplayLevels, $DriverPassed)
{
    $LogConfigs = @()
    $ValidLevels = $IntelLogConfigLevels.Keys
    $LogConfigCount = $DisplayModules.Count
    for ($i = 0; $i -lt $LogConfigCount; $i++)
    {
        if ($DriverPassed)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                SubModule  = "";
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
        else
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
    }

    return $LogConfigs
}

Function GetModuleName($ModuleId)
{
    return $IntelLogConfigModules.GetEnumerator().Where({$_.Value -eq $ModuleId}).Name
}

Function GetLevelName($LevelId)
{
    return $IntelLogConfigLevels.GetEnumerator().Where({$_.Value -eq $LevelId}).Name
}

Function GetDriverLogLevel($AdapterName)
{
    $DriverRegistryValue = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $DriverRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    }

    if ($null -ne $DriverRegistryValue)
    {
        $DriverRegistryValue = $DriverRegistryValue.DriverLogLevel
    }

    return $DriverRegistryValue
}

Function GetDriverLogConfig($DriverPassed, $SubModuleSum, $AdapterName, [ref]$ErrorMessages)
{
    $LogConfigs = @()
    $InactiveSubModuleArray = @()
    $ActiveSubModuleArray = @()

    $DriverLevel = GetDriverLogLevel $AdapterName
    if (-Not $DriverLevel)
    {
        # Set to Normal by default
        $DriverLevel = @($DRIVER_LOG_LEVEL_MAP.Keys)[1]
        if (-Not (SetDriverLogLevel $AdapterName $DriverLevel))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }
    }

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -eq $AdapterRegistryPath)
    {
        $ErrorMessages.Value += $Messages.NoCmdletSupport -f $AdapterName
        break
    }

    $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    if (-Not $SubModuleRegistryValue -or [string]::IsNullOrEmpty($SubModuleRegistryValue.SubModule))
    {
        # If driver passed but set has not been used, enable default driver logging
        if (-not (SetSubModuleRegistryValue $a $SUBMODULE_DEFAULT_SUM))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }

        $RegistrySubModuleSum = $SUBMODULE_DEFAULT_SUM
    }
    else
    {
        $RegistrySubModuleSum = $SubModuleRegistryValue.SubModule
    }

    foreach ($DisplayName in $SUB_MODULE_MAP.Keys)
    {
        if ($SUB_MODULE_MAP[$DisplayName] -band $SubModuleSum)
        {
            if ($SUB_MODULE_MAP[$DisplayName] -band $RegistrySubModuleSum)
            {
                $ActiveSubModuleArray += $DisplayName
            }
            else
            {
                $InactiveSubModuleArray += $DisplayName
            }
        }
    }

    if ($DriverPassed)
    {
        foreach ($ActiveSubModule in $ActiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $ActiveSubModule;
                                            Level       = $DriverLevel;
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
        foreach ($InactiveSubModule in $InactiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $InactiveSubModule;
                                            Level       = @($DRIVER_LOG_LEVEL_MAP.Keys)[-1]; # Disabled
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
    }
    else
    {
        $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                                        Name            = $AdapterName;
                                        Module          = $DRIVER_MODULE_NAME;
                                        SubModule       = @($ActiveSubModuleArray);
                                        Level           = $DriverLevel;
                                        ValidSubModules = $SUB_MODULE_MAP.Keys;
                                        ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
    }

    return $LogConfigs
}

Function GetSubModuleBitmask($SubModuleArray, [ref]$ErrorMessages)
{
    $iSubModuleSum = 0

    foreach ($SubModule in $SubModuleArray)
    {
        if ($SubModule -match "^0[xX][A-Fa-f0-9]+$" -or $SubModule -match "^[1-9]\d*$")
        {
            if ([int]$SubModule -le $SUM_SUBMODULES)
            {
                $iSubModuleSum = $iSubModuleSum -bor $SubModule
            }
            else
            {
                $iSubModuleSum = $SUM_SUBMODULES
                continue
            }
        }
        elseif ($SUB_MODULE_MAP.Keys -Contains $SubModule)
        {
            $iSubModuleSum = $iSubModuleSum -bor $SUB_MODULE_MAP[$SubModule]
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidSubModule -f $a, $SubModule
            $iSubModuleSum = -1
            break
        }
    }

    return $iSubModuleSum
}

Function GetLogmanGUID($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_WPP_GUID; break }
            'i40eb' { return $I40EB_WPP_GUID; break }
            'icea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $ICEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            'scea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $SCEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function GetLogName($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_LOGNAME; break }
            'i40eb' { return $I40EB_LOGNAME; break }
            'icea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $ICEA_LOGNAME
                }

                return $LogName
                break
            }
            'scea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $SCEA_LOGNAME
                }

                return $LogName
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function SetDriverLogLevel($AdapterName, $DriverLevel)
{
    $bSuccess = $false
    $SetDriverLogLevel = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetDriverLogLevel = Set-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -Type "String" -Value $DriverLevel -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetDriverLogLevel -and -Not [string]::IsNullOrEmpty($SetDriverLogLevel.DriverLogLevel))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function SetSubModuleRegistryValue($AdapterName, $SubModuleValue)
{
    $bSuccess = $false
    $SetSubModule = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetSubModule = Set-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -Value $SubModuleValue -Type "DWORD" -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetSubModule -and -Not [string]::IsNullOrEmpty($SetSubModule.SubModule))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function CreateDriverLogMetadataFile($AdapterName)
{
    $strDriverName = ($script:SupportedAdapters.Where({ $_.Name -eq $AdapterName })).Service

    $strOSSpecificDriverName = (GetOSSpecificDriverName $strDriverName)
    if (-Not $strOSSpecificDriverName)
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    $DriverFolder = GetDriverInfParentFolder $strOSSpecificDriverName "Start"
    $DriverMetaData = Get-Item "$DriverFolder\$strOSSpecificDriverName.sys" -ErrorAction SilentlyContinue

    if ($DriverMetaData)
    {
        $DriverVersionInfo = $DriverMetaData.VersionInfo
        $DriverLength = "$strOSSpecificDriverName.sys Length: " + $DriverMetaData.Length
        $DriverChecksum = Get-FileHash $DriverMetaData

        try
        {
            Set-Content $LogMetadataPath -Value $DriverVersionInfo -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverLength -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverChecksum -ErrorAction Stop
        }
        catch
        {
            $script:ErrorMessagesStart += $Messages.PathIncorrect
        }
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    return $true
}

Function ValidateMethodOutput($MethodOutput, $Verb)
{
    Write-Verbose -Message "MethodOutput: $MethodOutput verb: $Verb"

    do
    {
        $Result = $true
        if ($null -eq $MethodOutput)
        {
            switch ($Verb)
            {
                Get {$script:WarningMessagesGet += $Messages.NoCmdletSupport -f $a; break}
                Set {$script:WarningMessagesSet += $Messages.NoCmdletSupport -f $a; break}
                Start {$script:WarningMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
                Stop {$script:WarningMessagesStop += $Messages.NoCmdletSupport -f $a; break}
                Disable {$script:WarningMessagesDisable += $Messages.NoCmdletSupport -f $a; break}
                Reset {$script:WarningMessagesReset += $Messages.NoCmdletSupport -f $a; break}
            }

            $Result = $false
            break
        }

        if ($MethodOutput.OutStatus -ne 0)
        {
            switch ($Verb)
            {
                Get {$script:ErrorMessagesGet += $Messages.OperationFailed -f $a, $Verb; break}
                Set {$script:ErrorMessagesSet += $Messages.OperationFailed -f $a, $Verb; break}
                Start {$script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, $Verb; break}
                Stop {$script:ErrorMessagesStop += $Messages.OperationFailed -f $a, $Verb; break}
                Disable {$script:ErrorMessagesDisable += $Messages.OperationFailed -f $a, $Verb; break}
                Reset {$script:ErrorMessagesReset += $Messages.OperationFailed -f $a, $Verb; break}
            }

            $Result = $false
        }
    } while ($false)

    Return $Result
}

Function ValidateGetModuleParam([ref]$DisplayModules, [ref]$DisplayLevels, $ModuleNames, $MethodOutput, $AdapterName)
{
    if ($ModuleNames)
    {
        foreach ($ModuleName in $ModuleNames)
        {
            $ModuleIds = $null
            $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
            $ModuleIDCount = $ModuleIDNames.Count

            if ($ModuleIDCount -gt 0)
            {
                $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })

                if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
                {
                    $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                    foreach ($ModuleId in $ModuleIds)
                    {
                        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
                        {
                            if ($ModuleId -eq $MethodOutput.ModuleID[$i])
                            {
                                $DisplayModules.Value += GetModuleName($ModuleId)
                                $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
                                break
                            }
                        }
                    }
                }
            }
            else
            {
                #if user passes in an array of modules and one (or more) is spelled incorrectly,
                #show error msg for that 1 and continue on getting the modules for the rest
                $script:ErrorMessagesGet += $Messages.InvalidModule -f $AdapterName, $ModuleName
            }
        }
    }
    else
    {
        GetAllModulesLevel $MethodOutput $DisplayModules $DisplayLevels
    }
}

Function GetAllModulesLevel($MethodOutput, [ref]$DisplayModules, [ref]$DisplayLevels)
{
    if ($MethodOutput)
    {
        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$i])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
            # driver returns an array of 64 but only 31 modules populated
            if ($i -eq 31)
            {
                break
            }
        }
    }
}

Function ValidateLevelParam([ref]$InputLevelId, [ref]$DisplayLevels, $LevelName, $Count, $AdapterName, $DriverPassed)
{
    do
    {
        $NewLevelId = $null
        $Result = $true

        $LevelMatchArray = $IntelLogConfigLevels.GetEnumerator().Where({$_.Name -like $LevelName})
        if ($LevelMatchArray)
        {
            $NewLevelId = @($LevelMatchArray.Value)
        }
        # Handle default case of "Normal" if Level is not passed
        elseif ($DriverPassed -and -not $LevelName)
        {
            $NewLevelId = @($IntelLogConfigLevels["Normal"])
        }

        if ($null -eq $NewLevelId -or 0 -eq $NewLevelId.Count)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevel -f $AdapterName, $LevelName, $ModuleNames[0]
            $Result = $false
            break
        }

        if ($NewLevelId.count -gt 1)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevelWildcard
            $Result = $false
            break
        }

        for ($i = 0; $i -lt $Count; $i++)
        {
            $InputLevelId.Value[$i] = $NewLevelId[0]
            $DisplayLevels.Value += GetLevelName($NewLevelId[0])
        }

    } while ($false)

    Return $Result
}

Function ValidateSetModuleParam([ref]$InputModulesId, [ref]$DisplayModules, $ModuleNames, $AdapterName)
{
    $count = 0
    foreach ($ModuleName in $ModuleNames)
    {
        $ModuleIds = $null
        $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
        $ModuleIDCount = $ModuleIDNames.Count

        if ($ModuleIDCount -gt 0)
        {
            $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })
            if (-Not $ModuleIDNames -or $ModuleIDCount -gt $ModuleIDNames.Count)
            {
                $DisplayModules.Value += $DRIVER_MODULE_NAME
            }

            if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
            {
                $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                foreach ($ModuleId in $ModuleIds)
                {
                    $InputModulesId.Value[$count] = $ModuleId
                    $DisplayModules.Value += GetModuleName($ModuleId)
                    $count++
                }
            }
        }
        else
        {
            #if user passes in an array of modules and one (or more) is spelled incorrectly,
            #show error msg for that 1 and continue on getting the modules for the rest
            $script:ErrorMessagesSet += $Messages.InvalidModule -f $AdapterName, $ModuleName
        }
    }
}

Function LogmanProcessRunning($AdapterName, $LogName, $GUID)
{
    $Result = $false

    $LogmanQueryString = logman query $LogName -ets
    if ($LogmanQueryString -match ($GUID))
    {
        $script:ErrorMessagesStart += $Messages.LogmanRunning -f $AdapterName
        $Result = $true
    }

    return $Result
}

Function LogmanStart($LogmanPath, $LogName, $GUID, $SubModuleSum, $Level)
{
    if ($Append)
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level -a
    }
    else
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level
    }
}

Function LogmanStop($LogName)
{
    $null = logman stop $LogName -ets
}


Function ValidateLogmanStart($AdapterName)
{
    $Result = $true
    # -2147024629, -2147024773, -2147024893, -2147024891 error codes mean directory name is invalid
    # or filename, directory name, or volume label syntax is incorrect or permission is denied.
    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2147024629 -or $LASTEXITCODE -eq -2147024773 -or $LASTEXITCODE -eq -2147024893 -or $LASTEXITCODE -eq -2147024891)
    {
        $script:ErrorMessagesStart += $Messages.PathIncorrect
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, "Start"
        $Result = $false
    }
    return $Result
}

Function ValidateLogmanStop($AdapterName, [ref]$ErrorMessages)
{
    $Result = $true

    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2144337918)
    {
        $ErrorMessages.Value += $Messages.NoLogsStarted -f $AdapterName
        $Result = $false
    }
    else
    {
        $ErrorMessages.Value += $Messages.OperationFailed -f $AdapterName, "Stop"
        $Result = $false
    }
    Return $Result
}

Function ValidateConfigurationParam($ConfigurationNames, $ModuleNames, [ref]$DisplayConfigNames, $Verb, $a)
{
    $Result = $true

    do
    {
        if ($ConfigurationNames)
        {
            if ($ModuleNames)
            {
                switch ($Verb)
                {
                    Set { $script:ErrorMessagesSet += $Messages.InvalidParamsConfigAndModule; break }
                    Get { $script:ErrorMessagesGet += $Messages.InvalidParamsConfigAndModule; break }
                }
                $Result = $false
                break
            }

            if ($IntelLogConfigurations.Count -eq 0)
            {
                if ($Verb -eq "Set")
                {
                    $script:ErrorMessagesSet += $Messages.NoConfigurationSupport -f $a
                }
                elseif ($Verb -eq "Get")
                {
                    $script:WarningMessagesGet += $Messages.NoConfigurationSupport -f $a
                }

                $Result = $false
                break
            }

            foreach ($ConfigName in $ConfigurationNames)
            {
                $TmpDisplayConfigNames = $IntelLogConfigurations.Keys -Like $ConfigName

                if ($TmpDisplayConfigNames.Count -gt 0)
                {
                    $DisplayConfigNames.Value += $TmpDisplayConfigNames
                }
                else
                {
                    #if user passes in an array of configs and one (or more) is spelled incorrectly,
                    #show error msg for that 1 and continue on getting the config for the rest
                    switch ($Verb)
                    {
                        Set { $script:ErrorMessagesSet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                        Get { $script:ErrorMessagesGet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                    }

                    $Result = $false
                }
            }
        }
    } while ($false)

    return $Result
}

Function GetConfigLogs($DisplayConfigNames, [ref]$DisplayModules, [ref]$DisplayLevels, $MethodOutput)
{
    $ConfigurationID = $IntelLogConfigurations[$DisplayConfigNames]
    foreach ($ConfigID in $ConfigurationID)
    {
        $params = @{ConfigurationID = [uint32]$ConfigID}
        $GetConfigMethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevelsForConfiguration" $params

        for ($i = 0; $i -lt $GetConfigMethodOutput.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$GetConfigMethodOutput.ModuleID[$i]])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$GetConfigMethodOutput.ModuleID[$i]])
        }
    }
}

function GetConfigHashTable($AdapterName, $ConfigPath)
{
    $ConfigHashTable = @{}

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $ConfigPath
    if ($null -ne $FWLogConfigPath)
    {
        $LogConfigs = Get-Item $FWLogConfigPath -ErrorAction SilentlyContinue
        if ($null -ne $LogConfigs)
        {
            $LogConfigs = $LogConfigs.Property
            foreach ($Config in $LogConfigs)
            {
                $ConfigHashTable.add($Config, (Get-ItemProperty -Path $FWLogConfigPath -Name $Config -ErrorAction SilentlyContinue).$Config)
            }
        }
    }

    return $ConfigHashTable
}

function GetAdapterRegistryConfigPath($AdapterName, $Path)
{
    $ConfigPath = $null
    $AdapterObject = $script:MSNetAdapters.Where({$_.ifDesc -eq $AdapterName})
    $972Key = @(Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -Name "NetCfgInstanceId" -ErrorAction SilentlyContinue)
    $Adapter972Key = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterObject.InterfaceGuid})
    if ($Adapter972Key)
    {
        $ConfigPath = $Adapter972Key.PSPath + $Path
    }

    return $ConfigPath
}

function GetItemPropertyValue($AdapterName, $Path, $PropertyName)
{
    $PropertyItem = $null

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $Path
    if ($null -ne $FWLogConfigPath)
    {
        $PropertyItem = Get-ItemProperty -Path $FWLogConfigPath -Name $PropertyName -ErrorAction SilentlyContinue
    }

    $PropertyValue = $null
    if ($null -ne $PropertyItem)
    {
        $PropertyValue = $PropertyItem.$PropertyName
    }

    return $PropertyValue
}

# SIG # Begin signature block
# MIIt8gYJKoZIhvcNAQcCoIIt4zCCLd8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCWHYOF1gGMOvcg
# 7t6MGE6DwANnu0CF3XTMI6S6JZ5sMKCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQC0WMhOLa9BaZ9kSX5iJ3F/MA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjQwMjI2MDAwMDAwWhcNMjUwMjI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDBCfpjptqBxrQLJGyUHE47EvbngKTbZ0xMZoUj
# CJVmRhCCzWtZeKwlwhuI3bJyq4sSeejZxY7IMjroOoditsPm5xYohctw0UO+j1Th
# L71qce9bigWpDFDBBqksK5+011j/XPA+kRu/gJBolI50N8tIHHsH31NzD09/sN7U
# V242zTBy0TnMwanTXLMux/kVJbIloWSHRn0wIZmGuWESmWDrsLQEtSIo4zyUlzvQ
# UmJrtHMmJc3Rw/5TE7rC9Zq4Yt6s+BNu8i5howcK7yEOtiw/sKIlbACFJqpp6EUT
# Kwi7RRLKkuoL7G/+50XrJlCQqDbYxQAm7Tc2oFBVZW9xf4gUz3f48iflabLvDmc0
# pVWgDF0OmX+SzsHf94GYG3slCw8JJKfU66TfJEModuiDPwfgA6ripNWdBHqaDoY7
# JQPt6T6wierKjp64ABBHwyYSD55RIMUm/w33oe0i44tAlvUTkujJzwUQKpjXQ9av
# FyA2VqPea77rc3yiCRNeGQTpyO0CAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBSC0NSIL647v94GegQBXPynnV+p
# cDAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBADRT5U3ne/vFqaxPbuhkYXvhfhBMHNi+fHEkOCjPhEyqOkeU
# 1a7bjucUQh+Jb6yOlEKzb8KppbyTDNZzlo5NpkpBtFimyPY/h7pboom7mDOFXS9/
# NUuESrc/niHOTRkTzipQa125g5y/hgZ8a6+XoMjKi/rWOvllhdyUiJi6KY9x5+IN
# nXSgYZu/7xnGge/UybwaVrCVqmirark7p8I3vPOmIQeeGupn7qyzFdiMK5EEpPUI
# uO4po7YGOTQDgpdPjUQGmmGqbkrGgvH2fT2W/Ti8IZSgBM+3i3Rtqo50gOTOe9py
# fG30f9aFUtFHFc9BAA3kvG+Xqr4MLOdFYgQRGFXNjN5IA6zc0admMuG8m/hVasJN
# p+ACnv8HeWID2O6oTGPhwHZkvfgqL05qEO6ZiThnzwWDukiduuceeYxIVqyYW253
# hzgZCKnjWVdDT3gUWoO2TJNR7sZuP/gP7I2hyotU8uRTl3SvlFfbaVGHj+xVqR1k
# taptv3zLnJYUhbTyNjGCG1UwghtRAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRALRYyE4tr0Fpn2RJfmIncX8wDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgNVssg8lWdvCmZ3eLXFcP
# 8hfiKa5lRYUdxSoTmu8mAW0wDQYJKoZIhvcNAQEBBQAEggGAO9as6hmgdTYaDNen
# +gCzh5oeFpgG9I5wYM3ShpGQrR8vCRJOysZVJtuWBM8AenstKtljDij0T7ilnYiY
# ugyyI6HNVpBgJ5EEjaLPHXQLT2DI/zBs8EYjoFed4woqNZV90j1iu6hP2N7jXVBI
# B0IuetpOfp5m11eUv4Y0882fnuA+bt37G5o1VuuQyW31ufhl0D8wBJFgj3ueXOY5
# t1/bfyLi7RZgVG9r0W61X9EkyeZqSGDT9Vkze8ac++W0mmG44/MHFEOaXJRmQM/0
# Pw56x8lAhuZIDkUUQ4FZCHLDYiPk9Jtu8OJHXSSOflLUHzAofn0kV0Q91wMre6aK
# LSNvhNSxQ+rLN8TeBaQ4NvR5gfUwWMveWh49LIu0vMKbtGUyVSPuFY0cfPaqH+qS
# 7mY+8YAzOOQm3TV+a8SJOoQgcpF0IFWJpvlW8oP2c8RgrKOja+VMMYwdIu/7Al3Z
# hvXkUjStwNDKu7T/seqy2Hdh8V5JFSxSWGebLwPxQWOIJ8veoYIYzjCCGMoGCisG
# AQQBgjcDAwExghi6MIIYtgYJKoZIhvcNAQcCoIIYpzCCGKMCAQMxDzANBglghkgB
# ZQMEAgIFADCB9AYLKoZIhvcNAQkQAQSggeQEgeEwgd4CAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQgYKCgev7/X7hTma33X7mjTFHKogJu9RqVOJRhK114
# ungCFQCQzFnNmZmXMrQipS2DW6IT3y6FehgPMjAyNDA5MTcyMTMxMDlaoHKkcDBu
# MQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMTAwLgYDVQQDEydTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1w
# aW5nIFNpZ25lciBSMzWgghL/MIIGXTCCBMWgAwIBAgIQOlJqLITOVeYdZfzMEtjp
# iTANBgkqhkiG9w0BAQwFADBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMSwwKgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5n
# IENBIFIzNjAeFw0yNDAxMTUwMDAwMDBaFw0zNTA0MTQyMzU5NTlaMG4xCzAJBgNV
# BAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxMDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgU2ln
# bmVyIFIzNTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAI3RZ/TBSJu9
# /ThJOk1hgZvD2NxFpWEENo0GnuOYloD11BlbmKCGtcY0xiMrsN7LlEgcyoshtP3P
# 2J/vneZhuiMmspY7hk/Q3l0FPZPBllo9vwT6GpoNnxXLZz7HU2ITBsTNOs9fhbdA
# Wr/Mm8MNtYov32osvjYYlDNfefnBajrQqSV8Wf5ZvbaY5lZhKqQJUaXxpi4TXZKo
# hLgxU7g9RrFd477j7jxilCU2ptz+d1OCzNFAsXgyPEM+NEMPUz2q+ktNlxMZXPF9
# WLIhOhE3E8/oNSJkNTqhcBGsbDI/1qCU9fBhuSojZ0u5/1+IjMG6AINyI6XLxM8O
# AGQmaMB8gs2IZxUTOD7jTFR2HE1xoL7qvSO4+JHtvNceHu//dGeVm5Pdkay3Et+Y
# Tt9EwAXBsd0PPmC0cuqNJNcOI0XnwjE+2+Zk8bauVz5ir7YHz7mlj5Bmf7W8SJ8j
# QwO2IDoHHFC46ePg+eoNors0QrC0PWnOgDeMkW6gmLBtq3CEOSDU8iNicwNsNb7A
# Bz0W1E3qlSw7jTmNoGCKCgVkLD2FaMs2qAVVOjuUxvmtWMn1pIFVUvZ1yrPIVbYt
# 1aTld2nrmh544Auh3tgggy/WluoLXlHtAJgvFwrVsKXj8ekFt0TmaPL0lHvQEe5j
# Hbufhc05lvCtdwbfBl/2ARSTuy1s8CgFAgMBAAGjggGOMIIBijAfBgNVHSMEGDAW
# gBRfWO1MMXqiYUKNUoC6s2GXGaIymzAdBgNVHQ4EFgQUaO+kMklptlI4HepDOSz0
# FGqeDIUwDgYDVR0PAQH/BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAww
# CgYIKwYBBQUHAwgwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEF
# BQcCARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEoGA1UdHwRD
# MEEwP6A9oDuGOWh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1Rp
# bWVTdGFtcGluZ0NBUjM2LmNybDB6BggrBgEFBQcBAQRuMGwwRQYIKwYBBQUHMAKG
# OWh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGlu
# Z0NBUjM2LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20w
# DQYJKoZIhvcNAQEMBQADggGBALDcLsn6TzZMii/2yU/V7xhPH58Oxr/+EnrZjpIy
# vYTz2u/zbL+fzB7lbrPml8ERajOVbudan6x08J1RMXD9hByq+yEfpv1G+z2pmnln
# 5XucfA9MfzLMrCArNNMbUjVcRcsAr18eeZeloN5V4jwrovDeLOdZl0tB7fOX5F6N
# 2rmXaNTuJR8yS2F+EWaL5VVg+RH8FelXtRvVDLJZ5uqSNIckdGa/eUFhtDKTTz9L
# tOUh46v2JD5Q3nt8mDhAjTKp2fo/KJ6FLWdKAvApGzjpPwDqFeJKf+kJdoBKd2zQ
# uwzk5Wgph9uA46VYK8p/BTJJahKCuGdyKFIFfEfakC4NXa+vwY4IRp49lzQPLo7W
# ticqMaaqb8hE2QmCFIyLOvWIg4837bd+60FcCGbHwmL/g1ObIf0rRS9ceK4DY9rf
# BnHFH2v1d4hRVvZXyCVlrL7ZQuVzjjkLMK9VJlXTVkHpuC8K5S4HHTv2AJx6mOdk
# MJwS4gLlJ7gXrIVpnxG+aIniGDCCBhQwggP8oAMCAQICEHojrtpTaZYPkcg+XPTH
# 4z8wDQYJKoZIhvcNAQEMBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGlu
# ZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFUxCzAJ
# BgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMTI1Nl
# Y3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgQ0EgUjM2MIIBojANBgkqhkiG9w0B
# AQEFAAOCAY8AMIIBigKCAYEAzZjYQ0GrboIr7PYzfiY05ImM0+8iEoBUPu8mr4wO
# gYPjoiIz5vzf7d5wu8GFK1JWN5hciN9rdqOhbdxLcSVwnOTJmUGfAMQm4eXOls3i
# QwfapEFWuOsYmBKXPNSpwZAFoLGl5y1EaGGc5LByM8wjcbSF52/Z42YaJRsPXY54
# 5E3QAPN2mxDh0OLozhiGgYT1xtjXVfEzYBVmfQaI5QL35cTTAjsJAp85R+KAsOfu
# L9Z7LFnjdcuPkZWjssMETFIueH69rxbFOUD64G+rUo7xFIdRAuDNvWBsv0iGDPGa
# R2nZlY24tz5fISYk1sPY4gir99aXAGnoo0vX3Okew4MsiyBn5ZnUDMKzUcQrpVav
# GacrIkmDYu/bcOUR1mVBIZ0X7P4bKf38JF7Mp7tY3LFF/h7hvBS2tgTYXlD7TnIM
# PrxyXCfB5yQq3FFoXRXM3/DvqQ4shoVWF/mwwz9xoRku05iphp22fTfjKRIVpm4g
# FT24JKspEpM8mFa9eTgKWWCvAgMBAAGjggFcMIIBWDAfBgNVHSMEGDAWgBT2d2rd
# P/0BE/8WoWyCAi/QCj0UJTAdBgNVHQ4EFgQUX1jtTDF6omFCjVKAurNhlxmiMpsw
# DgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYI
# KwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMEwGA1UdHwRFMEMwQaA/oD2GO2h0
# dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ1Jv
# b3RSNDYuY3JsMHwGCCsGAQUFBwEBBHAwbjBHBggrBgEFBQcwAoY7aHR0cDovL2Ny
# dC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5nUm9vdFI0Ni5w
# N2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3
# DQEBDAUAA4ICAQAS13sgrQ41WAyegR0lWP1MLWd0r8diJiH2VVRpxqFGhnZbaF+I
# Q7JATGceTWOS+kgnMAzGYRzpm8jIcjlSQ8JtcqymKhgx1s6cFZBSfvfeoyigF8iC
# GlH+SVSo3HHr98NepjSFJTU5KSRKK+3nVSWYkSVQgJlgGh3MPcz9IWN4I/n1qfDG
# zqHCPWZ+/Mb5vVyhgaeqxLPbBIqv6cM74Nvyo1xNsllECJJrOvsrJQkajVz4xJwZ
# 8blAdX5umzwFfk7K/0K3fpjgiXpqNOpXaJ+KSRW0HdE0FSDC7+ZKJJSJx78mn+rw
# EyT+A3z7Ss0gT5CpTrcmhUwIw9jbvnYuYRKxFVWjKklW3z83epDVzoWJttxFpujd
# rNmRwh1YZVIB2guAAjEQoF42H0BA7WBCueHVMDyV1e4nM9K4As7PVSNvQ8LI1WRa
# TuGSFUd9y8F8jw22BZC6mJoB40d7SlZIYfaildlgpgbgtu6SDsek2L8qomG57Yp5
# qTqof0DwJ4Q4HsShvRl/59T4IJBovRwmqWafH0cIPEX7cEttS5+tXrgRtMjjTOp6
# A9l0D6xcKZtxnLqiTH9KPCy6xZEi0UDcMTww5Fl4VvoGbMG2oonuX3f1tsoHLaO/
# Fwkj3xVr3lDkmeUqivebQTvGkx5hGuJaSVQ+x60xJ/Y29RBr8Tm9XJ59AjCCBoIw
# ggRqoAMCAQICEDbCsL18Gzrno7PdNsvJdWgwDQYJKoZIhvcNAQEMBQAwgYgxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkg
# Q2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVV
# U0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTIxMDMyMjAw
# MDAwMFoXDTM4MDExODIzNTk1OVowVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1Nl
# Y3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFt
# cGluZyBSb290IFI0NjCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAIid
# 2LlFZ50d3ei5JoGaVFTAfEkFm8xaFQ/ZlBBEtEFAgXcUmanU5HYsyAhTXiDQkiUv
# pVdYqZ1uYoZEMgtHES1l1Cc6HaqZzEbOOp6YiTx63ywTon434aXVydmhx7Dx4IBr
# Aou7hNGsKioIBPy5GMN7KmgYmuu4f92sKKjbxqohUSfjk1mJlAjthgF7Hjx4vvyV
# DQGsd5KarLW5d73E3ThobSkob2SL48LpUR/O627pDchxll+bTSv1gASn/hp6IuHJ
# orEu6EopoB1CNFp/+HpTXeNARXUmdRMKbnXWflq+/g36NJXB35ZvxQw6zid61qmr
# lD/IbKJA6COw/8lFSPQwBP1ityZdwuCysCKZ9ZjczMqbUcLFyq6KdOpuzVDR3ZUw
# xDKL1wCAxgL2Mpz7eZbrb/JWXiOcNzDpQsmwGQ6Stw8tTCqPumhLRPb7YkzM8/6N
# nWH3T9ClmcGSF22LEyJYNWCHrQqYubNeKolzqUbCqhSqmr/UdUeb49zYHr7ALL8b
# AJyPDmubNqMtuaobKASBqP84uhqcRY/pjnYd+V5/dcu9ieERjiRKKsxCG1t6tG9o
# j7liwPddXEcYGOUiWLm742st50jGwTzxbMpepmOP1mLnJskvZaN5e45NuzAHteOR
# lsSuDt5t4BBRCJL+5EZnnw0ezntk9R8QJyAkL6/bAgMBAAGjggEWMIIBEjAfBgNV
# HSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQU9ndq3T/9ARP/
# FqFsggIv0Ao9FCUwDgYDVR0PAQH/BAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wEwYD
# VR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJMEcw
# RaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0
# aWZpY2F0aW9uQXV0aG9yaXR5LmNybDA1BggrBgEFBQcBAQQpMCcwJQYIKwYBBQUH
# MAGGGWh0dHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIB
# AA6+ZUHtaES45aHF1BGH5Lc7JYzrftrIF5Ht2PFDxKKFOct/awAEWgHQMVHol9ZL
# Syd/pYMbaC0IZ+XBW9xhdkkmUV/KbUOiL7g98M/yzRyqUOZ1/IY7Ay0YbMniIibJ
# rPcgFp73WDnRDKtVutShPSZQZAdtFwXnuiWl8eFARK3PmLqEm9UsVX+55DbVIz33
# Mbhba0HUTEYv3yJ1fwKGxPBsP/MgTECimh7eXomvMm0/GPxX2uhwCcs/YLxDnBdV
# VlxvDjHjO1cuwbOpkiJGHmLXXVNbsdXUC2xBrq9fLrfe8IBsA4hopwsCj8hTuwKX
# JlSTrZcPRVSccP5i9U28gZ7OMzoJGlxZ5384OKm0r568Mo9TYrqzKeKZgFo0fj2/
# 0iHbj55hc20jfxvK3mQi+H7xpbzxZOFGm/yVQkpo+ffv5gdhp+hv1GDsvJOtJinJ
# mgGbBFZIThbqI+MHvAmMmkfb3fTxmSkop2mSJL1Y2x/955S29Gu0gSJIkc3z30vU
# /iXrMpWx2tS7UVfVP+5tKuzGtgkP7d/doqDrLF1u6Ci3TpjAZdeLLlRQZm867eVe
# XED58LXd1Dk6UvaAhvmWYXoiLz4JA5gPBcz7J311uahxCweNxE+xxxR3kT0WKzAS
# o5G/PyDez6NHdIUKBeE3jDPs2ACc6CkJ1Sji4PKWVT0/MYIEkTCCBI0CAQEwaTBV
# MQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# EyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNgIQOlJqLITOVeYd
# ZfzMEtjpiTANBglghkgBZQMEAgIFAKCCAfkwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3
# DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yNDA5MTcyMTMxMDlaMD8GCSqGSIb3DQEJ
# BDEyBDDxJAIyCwDQVfKVAG4RsNSi88e8dq9N+9s/yHlXCYg3JHlWDt5pzaiG0tTp
# hWeRap0wggF6BgsqhkiG9w0BCRACDDGCAWkwggFlMIIBYTAWBBT4YJgZpvuILPfo
# UpfyoRlSGhZ3XzCBhwQUxq5U5HiG8Xw9VRJIjGnDSnr5wt0wbzBbpFkwVzELMAkG
# A1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2Vj
# dGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBSb290IFI0NgIQeiOu2lNplg+RyD5c
# 9MfjPzCBvAQUhT1jLZOCgmF80JA1xJHeksFC2scwgaMwgY6kgYswgYgxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0
# eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VS
# VHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5AhA2wrC9fBs656Oz3TbL
# yXVoMA0GCSqGSIb3DQEBAQUABIICAEJ2doBqdJw+K++b8DCtirm+rHjPNkeuuO46
# krJ5FnDi0FqOf6VfLE724eu0ltXndVjvb6G4fIAh2osi4gc0rPp+yuvf/omrUCug
# cmByCExmSY+yu2YR/Nqp021oevUHMEuaMGGE3KSgg1Y8lJTImXDivCc2Sv7sHTQG
# hhSDnVc4/hdCdn2PAixQxJlIuwwqz8/1r3S1PG6u1ajhJIlDEx0jX3juP5JmpULh
# CvwkukZsmPkhu6V4EFT4xMZ9As0ogr+YYEbKxmyZajVPA7QZecNVUEGG0g8z0sRz
# yy9KT/punc0g0FSRE6+we9R+XZbAxno5NzEt9SvmmSeD67ggPj1usFIAo/zpOLSs
# cTZY6jOUGUWrBfVvTIwrqqBA/JVuaF1f0A+NLt4RNZg8fOUy0tFlQBR/Wggwbbw5
# 6HKSNGR3TIKW9nrk3+fzqouRbBHmN0X8Ic8Ro6BjWfbZbfSQnkZKIjr3Jc+o6wiG
# DXTAUxeZAkRGgTMyFXtl72N0qokLB80CrfFp5NzzHGEJL2Rsh7wKUS3taNkWVs48
# V6+gayscOmh4oT2ifW6EYo69qiGK++8Zuns2rZGKNPvzrwaq3x6i0cFVXMtfqw9W
# MWJkhgiVWxwjXBE83nZMtYim8P4C3KaZIVj6InwMng0GELkRiuH5S78hzjULNg7y
# y/jk70SQ
# SIG # End signature block
